function geom = expplanarbend3D(geomparams)


%% PARAMETERS
%pre-bend straight length L1, post-bend straight length L2, bend radius and
% and angle R_B and theta_B, width increase ratio > 1, duct radius R_i
if ~exist('geomparams','var')
    geomparams.L1      = 0;
    geomparams.L2      = 0;
    geomparams.R_B     = 1;
    geomparams.theta_B = pi/2;
    geomparams.ratio   = 1;
    geomparams.R_i     = 0.5;
end

L1      = geomparams.L1;
L2      = geomparams.L2;
R_B     = geomparams.R_B;
theta_B = geomparams.theta_B;
s_o     = L1 + R_B*abs(theta_B) + L2;
ratio   = geomparams.ratio;
R_i     = geomparams.R_i;

geom.geomparams = geomparams;


%% SCALAR S-FUNCTIONS
    function R = ductradius(s)
        if s < s_o
            R = R_i*exp(log(ratio)*(s/s_o));
        else
            R = R_i*ratio;
        end
    end
    function Rdash = ductradiusderiv(s)
        if s < s_o
            Rdash = (R_i*log(ratio)/s_o)*exp(log(ratio)*(s/s_o));
        else
            Rdash = 0;
        end
    end
    function kappa = ductcurvature(s)
        if s < L1
            kappa = 0;
        elseif s > L1 + R_B*abs(theta_B)
            kappa = 0;
        else
            kappa = 1/R_B;
        end
    end
geom.L      = s_o;
geom.R      = @(s) ductradius(s);
geom.Rdash  = @(s) ductradiusderiv(s);
geom.kappa  = @(s) ductcurvature(s);
geom.tau    = @(s) 0;
geom.theta0 = @(s) 0;
geom.qyvar  = true;
geom.qzvar  = false;
geom.dim    = 3;


%% VECTOR S-FUNCTIONS
    function x = ductpathx(s,parity,cl)
        if s < L1
            x = cl*s;
        elseif s > L1 + R_B*abs(theta_B)
            x = cl*(L1 + cos(theta_B)*(s - L1 - R_B*abs(theta_B)))...
            + parity*sin(abs(theta_B));
        else
            x = cl*L1 + parity*sin((s - L1)/R_B);
        end
    end
    function y = ductpathy(s,parity,cl)
        if s < L1
            y = (1 - cl)*sign(theta_B);
        elseif s > L1 + R_B*abs(theta_B)
            y = cl*(R_B*sign(theta_B) ...
                + sin(theta_B)*(s - L1 - R_B*abs(theta_B)))...
                - parity*cos(theta_B)*sign(theta_B);
        else
            y = (cl*R_B - parity*cos((s - L1)/R_B))*sign(theta_B);
        end
    end
    function x = innerductpathx(s,parity,cl)
        if s < L1
            x = -cl*(L2 + cos(theta_B)*(L1 - s))...
            - parity*sin(abs(theta_B));
        elseif s > L1 + R_B*abs(theta_B)
            x = cl*(s - L1 - R_B*abs(theta_B) - L2);
        else
            x = -cl*L2 - parity*sin((L1 + R_B*abs(theta_B) - s)/R_B);
        end
    end
    function y = innerductpathy(s,parity,cl)
        if s < L1
            y = cl*(R_B + sin(abs(theta_B))*(L1 - s))...
                - parity*cos(theta_B);
        elseif s > L1 + R_B*abs(theta_B)
            y = (1 - cl);
        else
            y = (cl*R_B - parity*cos((L1 + R_B*abs(theta_B) - s)/R_B));
        end
    end

geom.qx = @(s) ductpathx(s,R_B,1);
geom.qy = @(s) ductpathy(s,R_B,1);
geom.qz = @(s) 0;
geom.nx = @(s) ductpathx(s,-1,0);
geom.ny = @(s) ductpathy(s,-1,0);
geom.nz = @(s) 0;
geom.bx = @(s) 0;
geom.by = @(s) 0;
geom.bz = @(s) sign(theta_B);


geom.innerqx = @(s) innerductpathx(s,R_B,1);
geom.innerqy = @(s) innerductpathy(s,R_B,1);
geom.innernx = @(s) innerductpathx(s,-1,0);
geom.innerny = @(s) innerductpathy(s,-1,0);


%% GEOMETRY NAME
geom.name = 'expplanarbend3D';

end