function grids = sthetaframe(geom,numparams,plotparams,Pchoice)

% returns plotting grids and spatially evaluated pressure amplitudes for a
% curved surface of varying s and theta

%% DUCT GEOMETRY
R_func  = geom().R;
theta_0 = geom().theta0;
qx      = geom().qx;
qy      = geom().qy;
qz      = geom().qz;
nx      = geom().nx;
ny      = geom().ny;
nz      = geom().nz;
bx      = geom().bx;
by      = geom().by;
bz      = geom().bz;


%% NUMERICAL DATA
alpha_mx    = numparams.alpha_mx;
a_mx        = numparams.a_mx;
modes       = numparams.modes;
C_alpha     = modes(:,6)';


%% PLOTTING PARAMETERS
r_0         = plotparams.r_0;
n_r         = plotparams.n_r;
n_s         = plotparams.n_s;
n_theta     = plotparams.n_theta;
s1          = plotparams.s1;
s2          = plotparams.s2;


%% PLOTTING GRID
%flat surface grids
R1          = R_func(s1);
R2          = R_func(s2);
R_pl01      = r_0*R1;
R_pl02      = r_0*R2;
r_pl01      = 0:R_pl01/n_r:R_pl01;
r_pl02      = 0:R_pl02/n_r:R_pl02;
theta_pl0   = 0:2*pi/n_theta:2*pi;
xx_pl01     = zeros(n_r+1,n_theta+1);
xx_pl02     = zeros(n_r+1,n_theta+1);
yy_pl01     = zeros(n_r+1,n_theta+1);
yy_pl02     = zeros(n_r+1,n_theta+1);
zz_pl01     = zeros(n_r+1,n_theta+1);
zz_pl02     = zeros(n_r+1,n_theta+1);

for I_r = 1:n_r+1
    for I_theta = 1:n_theta+1
        xx_pl01(I_r,I_theta) = qx(s1) ...
            + r_pl01(I_r)*cos(theta_pl0(I_theta) - theta_0(s1))*nx(s1) ...
            + r_pl01(I_r)*sin(theta_pl0(I_theta) - theta_0(s1))*bx(s1);
        xx_pl02(I_r,I_theta) = qx(s2) ...
            + r_pl02(I_r)*cos(theta_pl0(I_theta) - theta_0(s2))*nx(s2) ...
            + r_pl02(I_r)*sin(theta_pl0(I_theta) - theta_0(s2))*bx(s2);
        yy_pl01(I_r,I_theta) = qy(s1) ...
            + r_pl01(I_r)*cos(theta_pl0(I_theta) - theta_0(s1))*ny(s1) ...
            + r_pl01(I_r)*sin(theta_pl0(I_theta) - theta_0(s1))*by(s1);
        yy_pl02(I_r,I_theta) = qy(s2) ...
            + r_pl02(I_r)*cos(theta_pl0(I_theta) - theta_0(s2))*ny(s2) ...
            + r_pl02(I_r)*sin(theta_pl0(I_theta) - theta_0(s2))*by(s2);
        zz_pl01(I_r,I_theta) = qz(s1) ...
            + r_pl01(I_r)*cos(theta_pl0(I_theta) - theta_0(s1))*nz(s1) ...
            + r_pl01(I_r)*sin(theta_pl0(I_theta) - theta_0(s1))*bz(s1);
        zz_pl02(I_r,I_theta) = qz(s2) ...
            + r_pl02(I_r)*cos(theta_pl0(I_theta) - theta_0(s2))*nz(s2) ...
            + r_pl02(I_r)*sin(theta_pl0(I_theta) - theta_0(s2))*bz(s2);
    end
end

%curved surface grid
s_pl1       = s1:(s2 - s1)/n_s:s2;
theta_pl1   = 0:2*pi/n_theta:2*pi;
R_pl1       = zeros(n_s,1);
theta_0_pl1 = zeros(n_s,1);
for I_s = 1:n_s+1
    R_pl1(I_s)          = R_func(s_pl1(I_s))*r_0;
    theta_0_pl1(I_s)    = theta_0(s_pl1(I_s));
end
xx_pl1  = zeros(n_s+1,n_theta+1);
yy_pl1  = zeros(n_s+1,n_theta+1);
zz_pl1  = zeros(n_s+1,n_theta+1);

for I_s = 1:n_s+1
    for I_theta = 1:n_theta+1
        xx_pl1(I_s,I_theta) = qx(s_pl1(I_s)) ...
            + R_pl1(I_s)*cos(theta_pl1(I_theta) - theta_0_pl1(I_s))...
            *nx(s_pl1(I_s)) ...
            + R_pl1(I_s)*sin(theta_pl1(I_theta) - theta_0_pl1(I_s))...
            *bx(s_pl1(I_s));
        yy_pl1(I_s,I_theta) = qy(s_pl1(I_s)) ...
            + R_pl1(I_s)*cos(theta_pl1(I_theta) - theta_0_pl1(I_s))...
            *ny(s_pl1(I_s)) ...
            + R_pl1(I_s)*sin(theta_pl1(I_theta) - theta_0_pl1(I_s))...
            *by(s_pl1(I_s));
        zz_pl1(I_s,I_theta) = qz(s_pl1(I_s)) ...
            + R_pl1(I_s)*cos(theta_pl1(I_theta) - theta_0_pl1(I_s))...
            *nz(s_pl1(I_s)) ...
            + R_pl1(I_s)*sin(theta_pl1(I_theta) - theta_0_pl1(I_s))...
            *bz(s_pl1(I_s));
    end
end


%% INTERPOLANT EVALUATION
s_cell_pl01     = {s1, 1:(2*a_mx+1)*(alpha_mx+1)};
s_cell_pl02     = {s2, 1:(2*a_mx+1)*(alpha_mx+1)};
s_cell_pl1      = {s_pl1, 1:(2*a_mx+1)*(alpha_mx+1)};
P_alpha_a_pl01  = reshape(Pchoice(s_cell_pl01),alpha_mx+1,2*a_mx+1);
P_alpha_a_pl02  = reshape(Pchoice(s_cell_pl02),alpha_mx+1,2*a_mx+1);
P_a_pl1         = zeros(n_theta+1,2*a_mx+1,n_s+1);
P_alpha_a_pl1   = permute(reshape(Pchoice(s_cell_pl1),...
    size(Pchoice(s_cell_pl1),1),alpha_mx+1,2*a_mx+1),[3 2 1]);

%% MODE MULTIPLICATION
disp('summing over modes...')
J_alpha_pl01    = zeros(n_r+1,alpha_mx+1);
J_alpha_pl02    = zeros(n_r+1,alpha_mx+1);
cos_alpha_pl01  = zeros(n_theta+1,alpha_mx+1);
cos_alpha_pl02  = zeros(n_theta+1,alpha_mx+1);
psi_alpha_pl01  = zeros(alpha_mx+1,n_r+1,n_theta+1);
psi_alpha_pl02  = zeros(alpha_mx+1,n_r+1,n_theta+1);
psi_alpha_pl1   = zeros(alpha_mx+1,n_theta+1,n_s+1);

for I_alpha = 1:alpha_mx+1
    J_alpha_pl01(:,I_alpha) ...
        = besselj(modes(I_alpha,3),modes(I_alpha,2)*r_pl01/R1);
    J_alpha_pl02(:,I_alpha) ...
        = besselj(modes(I_alpha,3),modes(I_alpha,2)*r_pl02/R2);
    cos_alpha_pl01(:,I_alpha) ...
        = cos(modes(I_alpha,3)*(theta_pl0 - theta_0(s1)) ...
        - modes(I_alpha,5)*pi/2);
    cos_alpha_pl02(:,I_alpha) ...
        = cos(modes(I_alpha,3)*(theta_pl0 - theta_0(s2)) ...
        - modes(I_alpha,5)*pi/2);
    psi_alpha_pl01(I_alpha,:,:) = (1/(sqrt(pi)*R1))*C_alpha(I_alpha).*...
        J_alpha_pl01(:,I_alpha).*(cos_alpha_pl01(:,I_alpha))';
    psi_alpha_pl02(I_alpha,:,:) = (1/(sqrt(pi)*R2))*C_alpha(I_alpha).*...
        J_alpha_pl02(:,I_alpha).*(cos_alpha_pl02(:,I_alpha))';
    psi_alpha_pl1(I_alpha,:,:) = C_alpha(I_alpha)...
        *(besselj(modes(I_alpha,3),modes(I_alpha,2)*r_0))...
        *(cos(modes(I_alpha,3)*theta_pl1 - modes(I_alpha,5)*pi/2)'...
        .*cos(modes(I_alpha,3)*theta_0_pl1)'./(sqrt(pi)*R_pl1') ...
        + sin(modes(I_alpha,3)*theta_pl1 - modes(I_alpha,5)*pi/2)'...
        .*sin(modes(I_alpha,3)*theta_0_pl1)'./(sqrt(pi)*R_pl1'));
end
psi_alpha_pl01  = permute(reshape(permute(psi_alpha_pl01,[1 3 2]),...
    alpha_mx+1,(n_r+1)*(n_theta+1)),[2 1]);
psi_alpha_pl02  = permute(reshape(permute(psi_alpha_pl02,[1 3 2]),...
    alpha_mx+1,(n_r+1)*(n_theta+1)),[2 1]);
P_a_pl01        = psi_alpha_pl01*P_alpha_a_pl01;
P_a_pl02        = psi_alpha_pl02*P_alpha_a_pl02;

for I_s = 1:n_s+1
    P_a_pl1(:,:,I_s) = (P_alpha_a_pl1(:,:,I_s)*psi_alpha_pl1(:,:,I_s))';
end
P_a_pl1 =reshape(permute(P_a_pl1,[2 1 3]),[],(n_s+1)*(n_theta+1))';

xx.xx_pl01 = xx_pl01;
xx.xx_pl02 = xx_pl02;
xx.xx_pl1  = xx_pl1;

yy.yy_pl01 = yy_pl01;
yy.yy_pl02 = yy_pl02;
yy.yy_pl1  = yy_pl1;

zz.zz_pl01 = zz_pl01;
zz.zz_pl02 = zz_pl02;
zz.zz_pl1  = zz_pl1;

P_a.P_a_pl01 = P_a_pl01;
P_a.P_a_pl02 = P_a_pl02;
P_a.P_a_pl1 = P_a_pl1;


grids.xx = xx;
grids.yy = yy;
grids.zz = zz;
grids.P_a = P_a;


end